#include "options.h"

#include <iostream>
#include <ranges>
#include <algorithm>
#include <cmath>

#include "read.h"


std::tuple<bool, std::filesystem::path> getPDFPath(const std::string& str) {
    bool isPDF = false;
    std::filesystem::path pdfPath;
    if (std::filesystem::exists(std::filesystem::path(str))) {
        std::string fileExtension = std::filesystem::path(str).extension().string();

        std::ranges::transform(
            fileExtension, fileExtension.begin(),
            [](unsigned char c) { return std::tolower(c); }
            );

        if (fileExtension == ".pdf") {
            isPDF = true;
            pdfPath = std::filesystem::path(str);
        }
    }

    return std::make_tuple(isPDF, pdfPath);
}

std::expected<int, std::string> getDigitsFromString(const std::string& str) {
    std::string digitString = str
    | std::views::filter([](unsigned char c) { return std::isdigit(c); })
    | std::ranges::to<std::string>();

    if (!digitString.empty()) {
        return std::stoi(digitString);
    }

    return std::unexpected("There is no digital.");
}

std::tuple<int, int> getDpiResolution(const double& pageOriginalWidthPoints, const double& pageOriginalHeightPoints, const int& dpi) {
    double pageOriginalWidthInches = pageOriginalWidthPoints / 72;
    double pageOriginalHeightInches = pageOriginalHeightPoints / 72;
    int renderingWidthPixels = static_cast<int>(std::round(pageOriginalWidthInches * dpi));
    int renderingHeightPixels = static_cast<int>(std::round(pageOriginalHeightInches * dpi));

    return std::make_tuple(renderingWidthPixels, renderingHeightPixels);
}

std::expected<
    std::tuple<
        std::filesystem::path,
        std::vector<std::filesystem::path>,
        std::vector<std::array<int, 2>>,
        std::vector<int>,
        int
    >,
    std::string
>
getOptions(int argc, char** argv) {
    // All available options.
    std::vector<std::string> options = {
        "-o",  // the output folder path of images
        "-d",  // the output dpi of images
        "-s",  // the output pixel resolution of images
        "-p",  // the page numbers of the PDF document
        "-e",  // the format of images
        "-q"   // the quality for JPG/JPEG format
    };

    std::filesystem::path pdfPath;
    std::filesystem::path outputFolderPath;
    std::vector<std::filesystem::path> pageImagePaths;
    int imageDpi = 96;
    std::array<std::string, 2> imageSize;
    std::vector<std::array<int, 2>> imageSizes;
    std::vector<int> pageIndexes;
    std::string imageFormat;
    int imageQuality = 95;

    std::vector<std::string> filtered_options;

    // Distinguish all arguments.
    for (int i = 1; i < argc; ++i) {
        std::string arg = argv[i];

        auto recognizePDF = getPDFPath(arg);

        if (arg.size() >= 2 && std::ranges::contains(options, arg.substr(0, 2))) {
            filtered_options.emplace_back(arg.substr(0, 2));

            if (arg.size() > 2) {
                filtered_options.emplace_back(arg.substr(2));
            }
        } else if (!filtered_options.empty() && !std::get<0>(recognizePDF)) {
            std::string& lastArg = filtered_options.back();

            if (std::ranges::contains(options, lastArg)) {
                filtered_options.emplace_back(arg);
            } else {
                if (lastArg[lastArg.size() - 1] == ',') {
                    filtered_options.back() += arg;
                } else {
                    filtered_options.back() += ',';
                    filtered_options.back() += arg;
                }
            }
        } else if (std::get<0>(recognizePDF)) {
            filtered_options.emplace_back(arg);
            pdfPath = std::get<1>(recognizePDF);
        }
    }

    // If the path of the PDF file is invalid, the program should exit.
    if (pdfPath.empty()) {
        return std::unexpected("Please check the path of the PDF file, because there is no PDF file.");
    }

    int pageCount = getPDFPageCount(pdfPath);

    if (pageCount < 1) {
        return std::unexpected("Please check the PDF file, because there is no valid PDF file.");
    }

    // Remove invalid options.
    for (int i = 0; i < filtered_options.size(); ++i) {
        if (std::ranges::contains(options, filtered_options[i])
            && std::ranges::contains(options, filtered_options[i + 1])
            ) {
            filtered_options[i] = "";
        }
    }
    std::erase(filtered_options, "");

    // Recognize options.
    for (std::string& option: options) {
        auto viewResult = filtered_options
        | std::views::drop_while([option](auto& filteredOption){ return filteredOption != option; })
        | std::views::drop(1)
        | std::views::take(1);

        if (viewResult.empty()) {
            continue;
        }

        auto& viewResultValue = *viewResult.begin();

        if (option == "-o") {
            outputFolderPath = viewResultValue;
        }

        else if (option == "-d") {
            auto getDigits = getDigitsFromString(viewResultValue);
            if (getDigits.has_value()) {
                imageDpi = getDigits.value();
            }
        }

        else if (option == "-s") {
            auto imageSizeStrings = viewResultValue
                | std::views::split('x')
                | std::ranges::to<std::vector<std::string>>();

            std::vector<std::string> imageSizeDigitStrings;
            for (std::string& imageSizeString : imageSizeStrings) {
                auto getDigits = getDigitsFromString(imageSizeString);
                if (getDigits.has_value()) {
                    imageSizeDigitStrings.emplace_back(std::to_string(getDigits.value()));
                } else {
                    imageSizeDigitStrings.emplace_back();
                }
            }

            if (imageSizeDigitStrings.size() == 1 && !imageSizeDigitStrings[0].empty()) {
                imageSize = {imageSizeDigitStrings[0], ""};
            }

            else if (imageSizeDigitStrings.size() == 2) {
                if (!imageSizeDigitStrings[0].empty() && !imageSizeDigitStrings[1].empty()) {
                    imageSize = {imageSizeDigitStrings[0], imageSizeDigitStrings[1]};
                } else if (!imageSizeDigitStrings[0].empty()) {
                    imageSize = {imageSizeDigitStrings[0], ""};
                } else if (!imageSizeDigitStrings[1].empty()) {
                    imageSize = {"", imageSizeDigitStrings[1]};
                }
            }
        }

        else if (option == "-p") {
            std::stringstream stringStream(viewResultValue);
            std::string stringValue;

            // Page numbers are written like "1, 2, 3-5, -8, 10-" are acceptable.
            while (std::getline(stringStream, stringValue, ',')) {
                auto pageRangeStrings = stringValue
                    | std::views::split('-')
                    | std::ranges::to<std::vector<std::string>>();

                std::erase(pageRangeStrings, "");

                std::vector<int> pageRangeNumbers;
                for (std::string& pageRangeString: pageRangeStrings) {
                    auto getRangeDigits = getDigitsFromString(pageRangeString);
                    if (getRangeDigits.has_value()) {
                        int rangeDigits = getRangeDigits.value();
                        if (rangeDigits > 0 && rangeDigits <= pageCount) {
                            pageRangeNumbers.emplace_back(rangeDigits);
                        }
                    }
                }

                if (pageRangeNumbers.empty() || pageRangeNumbers.size() > 2) {
                    continue;
                }

                std::ranges::iota_view<int, int> pageNumbers;

                if (pageRangeNumbers.size() == 1) {
                    if (!std::ranges::contains(stringValue, '-')) {
                        pageIndexes.emplace_back(pageRangeNumbers[0]);
                    }

                    else if (stringValue[0] == '-') {
                        if (pageIndexes.empty()) {
                            pageNumbers = std::views::iota(1, pageRangeNumbers[0] + 1);
                        }

                        else if (pageRangeNumbers[0] > std::ranges::max(pageIndexes)) {
                            pageNumbers = std::views::iota(std::ranges::max(pageIndexes) + 1, pageRangeNumbers[0] + 1);
                        }
                    }

                    else if (stringValue[stringValue.size() - 1] == '-') {
                        if (pageIndexes.empty() || pageRangeNumbers[0] > std::ranges::max(pageIndexes)) {
                            pageNumbers = std::views::iota(pageRangeNumbers[0], pageCount + 1);
                        }

                        else if (pageRangeNumbers[0] == std::ranges::max(pageIndexes)) {
                            pageNumbers = std::views::iota(pageRangeNumbers[0] + 1, pageCount + 1);
                        }
                    }
                }

                if (pageRangeNumbers.size() == 2) {
                    if (pageRangeNumbers[0] < pageRangeNumbers[1]) {
                        if (pageIndexes.empty() || pageRangeNumbers[0] > std::ranges::max(pageIndexes)) {
                            pageNumbers = std::views::iota(pageRangeNumbers[0], pageRangeNumbers[1] + 1);
                        }

                        else if (pageRangeNumbers[0] == std::ranges::max(pageIndexes)) {
                            pageNumbers = std::views::iota(pageRangeNumbers[0] + 1, pageRangeNumbers[1] + 1);
                        }
                    }
                }

                if (!pageNumbers.empty()) {
                    for (int pageNumber : pageNumbers) {
                        pageIndexes.emplace_back(pageNumber);
                    }
                }
            }
        }

        else if (option == "-e") {
            imageFormat = viewResultValue;
            std::ranges::transform(imageFormat, imageFormat.begin(), [](unsigned char c) { return std::tolower(c); });
        }

        else if (option == "-q") {
            auto getDigits = getDigitsFromString(viewResultValue);
            if (getDigits.has_value()) {
                imageQuality = getDigits.value();
                if (imageQuality < 1 || imageQuality > 100) {
                    imageQuality = 95;
                }
            }
        }
    }

    if (outputFolderPath.empty()) {
        outputFolderPath = pdfPath.parent_path();
    }

    if (pageIndexes.empty()) {
        for (int pageNumber : std::views::iota(1, pageCount + 1)) {
            pageIndexes.emplace_back(pageNumber);
        }
    }

    std::ranges::sort(pageIndexes);

    std::vector<std::string> imageFormats = {"jpg", "jpeg", "png"};
    if (!std::ranges::contains(imageFormats, imageFormat)) {
        imageFormat = "png";
    }

    auto [imageExpectedWidthPixels, imageExpectedHeightPixels] = imageSize;

    auto pageOriginalSizes = getPageOriginalSizes(pdfPath, pageIndexes);

    for (int i = 0 ; i < pageIndexes.size(); ++i) {
        int pageIndex = pageIndexes[i];
        std::cout << pageIndex << " ";

        auto [pageOriginalWidthPoints, pageOriginalHeightPoints] = pageOriginalSizes[i];

        double pageAspectRatio = pageOriginalWidthPoints / pageOriginalHeightPoints;

        auto dpiResolution = getDpiResolution(pageOriginalWidthPoints, pageOriginalHeightPoints, imageDpi);

        int renderingWidthPixels = std::get<0>(dpiResolution);
        int renderingHeightPixels = std::get<1>(dpiResolution);

        if (!imageExpectedWidthPixels.empty() && imageExpectedHeightPixels.empty()) {
            renderingWidthPixels = std::stoi(imageExpectedWidthPixels);
            renderingHeightPixels = static_cast<int>(std::round(renderingWidthPixels / pageAspectRatio));
        }

        else if (imageExpectedWidthPixels.empty() && !imageExpectedHeightPixels.empty()) {
            renderingHeightPixels = std::stoi(imageExpectedHeightPixels);
            renderingWidthPixels = static_cast<int>(std::round(renderingHeightPixels * pageAspectRatio));
        }

        if (!imageExpectedWidthPixels.empty() && !imageExpectedHeightPixels.empty()) {
            renderingWidthPixels = std::stoi(imageExpectedWidthPixels);
            renderingHeightPixels = std::stoi(imageExpectedHeightPixels);
        }

        imageSizes.emplace_back(std::array{renderingWidthPixels, renderingHeightPixels});
    }

    std::cout << "\n\n";

    // Create a subdirectory for saving images, and decide the names of images.
    std::string pdfFileName = pdfPath.stem().string();
    std::string subdirectoryName = pdfFileName + "_pages";
    std::filesystem::path subdirectoryPath = outputFolderPath / subdirectoryName;
    std::filesystem::create_directories(subdirectoryPath);
    for (int& pageIndex : pageIndexes) {
        std::string imageFileFullName;

        imageFileFullName += pdfFileName + "_p";
        imageFileFullName += std::to_string(pageIndex);

        if (imageFormat == "jpeg" || imageFormat == "jpg") {
            imageFileFullName += ".jpg";
        } else {
            imageFileFullName += ".png";
        }

        pageImagePaths.emplace_back(subdirectoryPath / imageFileFullName);
    }

    return std::make_tuple(
        pdfPath, pageImagePaths, imageSizes, pageIndexes, imageQuality
        );
}