#include "read.h"

#include <cstring>

#include "fpdfview.h"


int getPDFPageCount(const std::filesystem::path& pdfPath) {
    int pageCount = 0;

    FPDF_InitLibrary();
    FPDF_DOCUMENT pdfDocument = FPDF_LoadDocument(pdfPath.c_str(), nullptr);

    pageCount = FPDF_GetPageCount(pdfDocument);

    FPDF_CloseDocument(pdfDocument);
    FPDF_DestroyLibrary();

    return pageCount;
}

std::vector<std::array<double, 2>> getPageOriginalSizes(const std::filesystem::path& pdfPath, const std::vector<int>& pageIndexes) {
    std::vector<std::array<double, 2>> pageOriginalSizes;

    FPDF_InitLibrary();
    FPDF_DOCUMENT pdfDocument = FPDF_LoadDocument(pdfPath.c_str(), nullptr);

    for (int pageIndex : pageIndexes) {
        FPDF_PAGE pdfPage = FPDF_LoadPage(pdfDocument, pageIndex - 1);

        double pageOriginalWidthPoints = FPDF_GetPageWidth(pdfPage);
        double pageOriginalHeightPoints = FPDF_GetPageHeight(pdfPage);

        pageOriginalSizes.emplace_back(std::array{pageOriginalWidthPoints, pageOriginalHeightPoints});

        FPDF_ClosePage(pdfPage);
    }

    FPDF_CloseDocument(pdfDocument);
    FPDF_DestroyLibrary();

    return pageOriginalSizes;
}

std::tuple<std::vector<unsigned char>, int> getPDFPageBuffer(
    const std::filesystem::path& pdfPath,
    const int& pageActualIndex,
    const int& renderingWidthPixels,
    const int& renderingHeightPixels
    ) {
    FPDF_InitLibrary();
    FPDF_DOCUMENT pdfDocument = FPDF_LoadDocument(pdfPath.c_str(), nullptr);
    FPDF_PAGE pdfPage = FPDF_LoadPage(pdfDocument, pageActualIndex);

    FPDF_BITMAP bitmap = FPDFBitmap_Create(renderingWidthPixels, renderingHeightPixels, 0);
    FPDFBitmap_FillRect(bitmap, 0, 0, renderingWidthPixels, renderingHeightPixels, 0xFFFFFFFF);
    FPDF_RenderPageBitmap(
        bitmap, pdfPage, 0, 0, renderingWidthPixels, renderingHeightPixels, 0, 0
        );

    auto buffer = static_cast<unsigned char*>(FPDFBitmap_GetBuffer(bitmap));
    int stride = FPDFBitmap_GetStride(bitmap);

    size_t size = stride * renderingHeightPixels;
    std::vector<unsigned char> bufferBGRA(size);
    memcpy(bufferBGRA.data(), buffer, size);

    FPDFBitmap_Destroy(bitmap);

    FPDF_ClosePage(pdfPage);
    FPDF_CloseDocument(pdfDocument);
    FPDF_DestroyLibrary();

    return std::make_tuple(bufferBGRA, stride);
}