#include "save_pdf.h"

#include <iostream>
#include <sstream>
#include <iomanip>
#include <random>
#include <chrono>
#include <ctime>
#include <format>

#include "calculator.h"


bool SavePDF::createPDF(
    const std::vector<std::filesystem::path>& imagePaths,
    const std::filesystem::path& outputPDFPath,
    const std::array<int, 2>& paperSize,
    const int& outputWidth,
    const int& outputQuality,
    const int& imageNumberPerPage,
    const std::array<int, 4>& margins,
    const std::array<int, 2>& spacings
    ) {
    std::vector<HandleImage::ImageData> images;

    for (const auto& imagePath : imagePaths) {
        HandleImage::ImageData imageData;
        HandleImage::handleImage(imagePath, imageData, paperSize, outputWidth, outputQuality);
        images.push_back(imageData);
    }

    std::vector<std::vector<int>> imageIndexesForAllPages;
    for (int imageIndex = 0; imageIndex < images.size();) {
        std::vector<int> imageIndexes;
        for (int i = 0; i < imageNumberPerPage; ++i) {
            auto currentImageIndex = imageIndex + i;

            if (currentImageIndex < images.size()) {
                imageIndexes.emplace_back(currentImageIndex);
            }
        }

        imageIndexesForAllPages.emplace_back(imageIndexes);

        imageIndex += imageNumberPerPage;
    }

    int pageCount = static_cast<int>(imageIndexesForAllPages.size());

    /* Generate the IDs of objects: catalog -> pages -> page -> stream contents -> images -> info. */
    int currentObjectID = 1;
    std::vector<int> pageObjectIDs;
    std::vector<int> contentObjectIDs;
    std::vector<int> imageObjectIDs;

    // Generate the ID of catalog object.
    int catalogObjectID = currentObjectID++;

    // Generate the ID of pages object.
    int pagesInfoObjectID = currentObjectID++;

    // Generate the IDs of page objects.
    for (int pageIndex = 0; pageIndex < pageCount; ++pageIndex) {
        int objectID = currentObjectID++;
        pageObjectIDs.push_back(objectID);
    }

    // Generate the IDs of stream contents.
    for (int pageIndex = 0; pageIndex < pageCount; ++pageIndex) {
        int objectID = currentObjectID++;
        contentObjectIDs.push_back(objectID);
    }

    // Generate the IDs of image objects.
    for (auto& img : images) {
        int objectID = currentObjectID++;
        imageObjectIDs.push_back(objectID);
    }

    // Generate the ID of info object.
    int documentInfoObjectID = currentObjectID++;

    // Generate the ID of xref object.
    int xrefObjectID = currentObjectID++;

    /* Write objects to a PDF document. */
    std::ostringstream buffer;
    std::vector<size_t> objectOffsets;

    buffer << "%PDF-2.0" << "\r\n";
    buffer << "%\xE2\xE3\xCF\xD3" << "\r\n";

    // Generate and write catalog object.
    auto catalogInfo = generateCatalog(pagesInfoObjectID);
    addObject(buffer, objectOffsets, catalogObjectID, catalogInfo);

    // Generate and write pages object.
    auto pagesInfo = generatePagesInfo(pageObjectIDs);
    addObject(buffer, objectOffsets, pagesInfoObjectID, pagesInfo);

    // Generate and write page objects.
    for (int pageIndex = 0; pageIndex < pageCount; ++pageIndex) {
        auto paperSizePoints = getPaperSizePoints(
            images, imageIndexesForAllPages[pageIndex], paperSize, imageNumberPerPage
        );

        auto page = createPage(
            imageObjectIDs,
            contentObjectIDs,
            pagesInfoObjectID,
            imageIndexesForAllPages[pageIndex],
            paperSizePoints,
            pageIndex
        );

        addObject(buffer, objectOffsets, pageObjectIDs[pageIndex], page);
    }

    // Generate and write stream content objects.
    for (int pageIndex = 0; pageIndex < pageCount; ++pageIndex) {
        auto paperSizePoints = getPaperSizePoints(
            images, imageIndexesForAllPages[pageIndex], paperSize, imageNumberPerPage
        );

        std::vector<std::array<int, 2>> imageOutputPointSizes;
        for (auto imageIndex : imageIndexesForAllPages[pageIndex]) {
            auto imageWidth = images[imageIndex].width;
            auto imageHeight = images[imageIndex].height;

            auto imageWidthPoints = Calculator::convertPixelsToPoints(imageWidth, images[imageIndex].xDpi);
            auto imageHeightPoints = Calculator::convertPixelsToPoints(imageHeight, images[imageIndex].yDpi);

            std::array<int, 2> imagePointSize{imageWidthPoints, imageHeightPoints};
            imageOutputPointSizes.emplace_back(imagePointSize);
        }

        auto [topMargin, bottomMargin, leftMargin, rightMargin] = margins;
        auto [horizontalSpacing, verticalSpacing] = spacings;

        std::array<int, 4> pointMargins = {
            Calculator::convertMMToPoints(topMargin),
            Calculator::convertMMToPoints(bottomMargin),
            Calculator::convertMMToPoints(leftMargin),
            Calculator::convertMMToPoints(rightMargin)
        };

        std::array<int, 2> pointSpacings = {
            Calculator::convertMMToPoints(horizontalSpacing),
            Calculator::convertMMToPoints(verticalSpacing)
        };

        auto pageImageRects = Calculator::getObjectRectsInArea(
            paperSizePoints, imageOutputPointSizes, pointMargins, pointSpacings
        );

        auto content = createContentStream(imageIndexesForAllPages[pageIndex], pageImageRects);

        addObject(buffer, objectOffsets, contentObjectIDs[pageIndex], content);
    }

    // Generate and write image objects.
    for (int i = 0; i < images.size(); ++i) {
        std::string content;
        if (images[i].type == "jpeg") {
            content = encodeJPEG(images[i]);
        } else {
            content = encodePNG(images[i]);
        }

        addObject(buffer, objectOffsets, imageObjectIDs[i], content);
    }

    // Generate and write info object.
    auto documentInfo = generateDocumentInfo();
    addObject(buffer, objectOffsets, documentInfoObjectID, documentInfo);

    // Generate and write xref and trailer.
    generateXref(buffer, objectOffsets, xrefObjectID, catalogObjectID, documentInfoObjectID);

    std::ofstream out(outputPDFPath, std::ios::binary);
    out << buffer.str();

    if (!std::filesystem::exists(outputPDFPath)) {
        return false;
    }

    return true;
}

std::string SavePDF::generateCatalog(const int& pagesInfoObjectID) {
    std::ostringstream catalogInfo;
    catalogInfo << "<<" << "\r\n"
                << "/Type /Catalog" << "\r\n"
                << "/Pages " << pagesInfoObjectID << " 0 R" << "\r\n"
                << "/Version /2.0" << "\r\n"
                << ">>";

    return catalogInfo.str();
}

std::string SavePDF::generatePagesInfo(const std::vector<int>& pageObjectIDs) {
    std::ostringstream pagesInfo;
    pagesInfo << "<<" << "\r\n"
              << "/Type /Pages" << "\r\n"
              << "/Kids [";
    for (int pageObjectID : pageObjectIDs) {
        pagesInfo << " " << pageObjectID << " 0 R";
    }
    pagesInfo << " ]" << "\r\n"
              << "/Count " << pageObjectIDs.size() << "\r\n"
              << ">>";

    return pagesInfo.str();
}

std::array<int, 2> SavePDF::getPaperSizePoints(
    const std::vector<HandleImage::ImageData>& images,
    const std::vector<int>& imageIndexes,
    const std::array<int, 2>& paperSize,
    const int& imageNumberPerPage
    ) {
    /* The default size of paper is the size of an image. */
    auto& img = images[imageIndexes[0]];
    int paperWidthPoints = Calculator::convertPixelsToPoints(img.width, img.xDpi);
    int paperHeightPoints = Calculator::convertPixelsToPoints(img.height, img.yDpi);

    /* If a page should contain multiple images but paper size is not given,
     * A4 size is set by default.
     */
    if (imageNumberPerPage > 1 && paperSize[0] == 0 && paperSize[1] == 0) {
        paperWidthPoints = Calculator::convertMMToPoints(210);
        paperHeightPoints = Calculator::convertMMToPoints(297);
    }

    if (paperSize[0] > 0 && paperSize[1] > 0) {
        paperWidthPoints = Calculator::convertMMToPoints(paperSize[0]);
        paperHeightPoints = Calculator::convertMMToPoints(paperSize[1]);
    }

    return std::array{paperWidthPoints, paperHeightPoints};
}

std::string SavePDF::createPage(
    const std::vector<int>& imageObjectIDs,
    const std::vector<int>& contentObjectIDs,
    const int& pagesInfoObjectID,
    const std::vector<int>& imageIndexes,
    const std::array<int, 2>& paperSizePoints,
    const int& pageIndex
    ) {
    /* For /MediaBox, the unit of an image's width and height is point. */
    auto [paperWidthPoints, paperHeightPoints] = paperSizePoints;

    std::ostringstream xObject;
    for (int i = 0; i < imageIndexes.size(); ++i) {
        xObject << "/Im" << imageIndexes[i] << " " << imageObjectIDs[imageIndexes[i]] << " 0 R";

        if (imageIndexes.size() > 1 && i < imageIndexes.size() - 1) {
            xObject << "\r\n";
        }
    }

    std::ostringstream page;
    page << "<<" << "\r\n"
         << "/Type /Page" << "\r\n"
         << "/Parent " << pagesInfoObjectID << " 0 R" << "\r\n"
         << "/MediaBox [0 0 " << paperWidthPoints << " " << paperHeightPoints << "]" << "\r\n"
         << "/CropBox [0 0 " << paperWidthPoints << " " << paperHeightPoints << "]" << "\r\n"
         << "/Contents " << contentObjectIDs[pageIndex] << " 0 R" << "\r\n"
         << "/Resources <<" << "\r\n"
         << "/XObject << " << xObject.str() << " >>" << "\r\n"
         << ">>" << "\r\n"
         << ">>";

    return page.str();
}

std::string SavePDF::createContentStream(
    const std::vector<int>& imageIndexes,
    const std::vector<std::array<int, 4>>& imageRects
    ) {
    std::ostringstream contentStream;

    for (int i = 0; i < imageIndexes.size(); ++i) {
        /* For content stream, the unit of an image's width and height is point. */
        auto [
            imageCoordinateX,
            imageCoordinateY,
            imageOutputWidthPoints,
            imageOutputHeightPoints
            ] = imageRects[i];

        contentStream << 'q' << "\r\n"
                      << imageOutputWidthPoints << " 0 0 " << imageOutputHeightPoints << ' '
                      << imageCoordinateX << ' ' << imageCoordinateY << " cm" << "\r\n"
                      << "/Im" << imageIndexes[i] << " Do" << "\r\n"
                      << 'Q';

        if (imageIndexes.size() > 1 && i < imageIndexes.size() - 1) {
            contentStream << "\r\n";
        }
    }

    std::string contentStreamString = contentStream.str();

    std::ostringstream content;
    content << "<<" << "\r\n"
            << "/Length " << contentStreamString.size() << "\r\n"
            << ">>" << "\r\n"
            << "stream" << "\r\n"
            << contentStreamString << "\r\n"
            << "endstream";

    return content.str();
}

std::string SavePDF::encodeJPEG(const HandleImage::ImageData& imageData) {
    std::ostringstream imageXObject;

    /* For /Width and /Height, the unit is pixel. */
    imageXObject << "<<" << "\r\n"
                 << "/Type /XObject" << "\r\n"
                 << "/Subtype /Image" << "\r\n"
                 << "/Width " << imageData.width << "\r\n"
                 << "/Height " << imageData.height << "\r\n"
                 << "/ColorSpace /DeviceRGB" << "\r\n"
                 << "/BitsPerComponent 8" << "\r\n"
                 << "/Filter /DCTDecode" << "\r\n"
                 << "/Length " << imageData.data.size() << "\r\n"
                 << ">>" << "\r\n"
                 << "stream" << "\r\n";

    imageXObject.write(
        reinterpret_cast<const char*>(imageData.data.data()),
        static_cast<int>(imageData.data.size())
    );

    imageXObject << "\r\n"
                 << "endstream";

    return imageXObject.str();
}

std::string SavePDF::encodePNG(const HandleImage::ImageData& imageData) {
    std::ostringstream imageXObject;

    /* For /Width and /Height, the unit is pixel. */
    imageXObject << "<<" << "\r\n"
                 << "/Type /XObject" << "\r\n"
                 << "/Subtype /Image" << "\r\n"
                 << "/Width " << imageData.width << "\r\n"
                 << "/Height " << imageData.height << "\r\n"
                 << "/ColorSpace /DeviceRGB" << "\r\n"
                 << "/BitsPerComponent 8" << "\r\n"
                 << "/Filter /FlateDecode" << "\r\n"
                 << "/Length " << imageData.data.size() << "\r\n"
                 << ">>" << "\r\n"
                 << "stream" << "\r\n";

    imageXObject.write(
        reinterpret_cast<const char*>(imageData.data.data()),
        static_cast<int>(imageData.data.size())
    );

    imageXObject << "\r\n"
                 << "endstream";

    return imageXObject.str();
}

std::string SavePDF::generateDocumentInfo() {
    auto now = std::chrono::system_clock::now();
    auto utc_time = std::chrono::time_point_cast<std::chrono::seconds>(now);
    auto time_t_now = std::chrono::system_clock::to_time_t(utc_time);
    auto tm_utc = *std::gmtime(&time_t_now);

    std::string currentTime = std::format(
        "D:{:04}{:02}{:02}{:02}{:02}{:02}+00'00'",
        tm_utc.tm_year + 1900,
        tm_utc.tm_mon + 1,
        tm_utc.tm_mday,
        tm_utc.tm_hour,
        tm_utc.tm_min,
        tm_utc.tm_sec
    );

    std::ostringstream documentInfo;
    documentInfo << "<<" << "\r\n"
                 << "/Creator (imageToPDF)" << "\r\n"
                 << "/Producer (libImageToPDF)" << "\r\n"
                 << "/CreationDate (" << currentTime << ")" << "\r\n"
                 << "/ModDate (" << currentTime << ")" << "\r\n"
                 << ">>";

    return documentInfo.str();
}

std::string SavePDF::generatePDFID() {
    std::ostringstream idStream;
    idStream << std::hex << std::setfill('0');

    std::random_device rd;
    std::mt19937_64 gen(rd());
    std::uniform_int_distribution<uint64_t> distrib;

    idStream << std::setw(16) << distrib(gen);
    idStream << std::setw(16) << distrib(gen);

    return idStream.str();
}

void SavePDF::addObject(
    std::ostringstream& buffer,
    std::vector<size_t>& objectOffsets,
    const int& objectID,
    const std::string& content
    ) {
    const size_t objectOffset = buffer.tellp();
    objectOffsets.push_back(objectOffset);
    buffer << objectID << " 0 obj" << "\r\n"
           << content << "\r\n"
           << "endobj" << "\r\n";
}

void SavePDF::writeXrefStreamData(
    std::ostringstream& xrefStreamData,
    size_t value,
    const int& byteWidth
    ) {
    std::vector<char> xrefPiece(byteWidth);

    for (int i = byteWidth - 1; i >= 0; --i) {
        xrefPiece[i] = static_cast<char>(value & 0xFF);
        value >>= CHAR_BIT;
    }

    xrefStreamData.write(xrefPiece.data(), byteWidth);
}

void SavePDF::generateXref(
    std::ostringstream& buffer,
    const std::vector<size_t>& objectOffsets,
    const int& xrefObjectID,
    const int& catalogObjectID,
    const int& documentInfoObjectID
    ) {
    constexpr int typeWidth = 1;
    constexpr int offsetWidth = 5;
    constexpr int generationWidth = 2;

    std::ostringstream xrefStreamData;

    /* Generate a xref record for the first object, which ID is 0. */
    writeXrefStreamData(xrefStreamData, 0, typeWidth);
    writeXrefStreamData(xrefStreamData, 0, offsetWidth);
    writeXrefStreamData(xrefStreamData, 65535, generationWidth);

    /* Generate xref records for normal objects. */
    for (const auto& objectOffset : objectOffsets) {
        writeXrefStreamData(xrefStreamData, 1, typeWidth);
        writeXrefStreamData(xrefStreamData, objectOffset, offsetWidth);
        writeXrefStreamData(xrefStreamData, 0, generationWidth);
    }

    std::string xrefStreamString = xrefStreamData.str();

    /* Objects include the first object with ID 0, normal objects, and xref object. */
    int objectCount = static_cast<int>(objectOffsets.size() + 2);

    const size_t xrefStreamOffset = buffer.tellp();

    buffer << xrefObjectID << " 0 obj" << "\r\n"
           << "<<" << "\r\n"
           << "/Type /XRef" << "\r\n"
           << "/Size " << (objectCount) << "\r\n"
           << "/W [ " << typeWidth << " " << offsetWidth << " " << generationWidth << " ]" << "\r\n"
           << "/Index [0 " << xrefObjectID << "]" << "\r\n"
           << "/Root " << catalogObjectID << " 0 R" << "\r\n"
           << "/Info " << documentInfoObjectID << " 0 R" << "\r\n";

    auto pdfID = generatePDFID();

    buffer << "/ID [<" << pdfID << "> <" << pdfID << ">]" << "\r\n"
           << "/Length " << xrefStreamString.size() << "\r\n"
           << ">>" << "\r\n"
           << "stream" << "\r\n"
           << xrefStreamString << "\r\n"
           << "endstream" << "\r\n"
           << "endobj" << "\r\n";

    buffer << "startxref" << "\r\n"
           << xrefStreamOffset << "\r\n"
           << "%%EOF" << "\r\n";
}