#include "options.h"

#include <ranges>
#include <algorithm>

#include "config.h"


std::string Options::selectProvider(const std::string& purpose) {
    auto supportedProviders = Config::supportedProviders;
    std::string selectedProvider;

    int i = 1;
    for (auto& provider : supportedProviders) {
        std::cout << i << ". " << provider << "  ";
        ++i;
    }

    std::cout << "\n";

    std::cout << "Please choose the number of the AI Provider that providers " << purpose << ".\n"
              << "(press 0 to exit without any selection)\n";

    int number;
    std::cin >> number;

    /* The method std::cin.ignore() is needed when using std::getline
     * after std::cin, because std::cin leaves \n and std::getline
     * accepts \n as the end of a line. Without std::cin.ignore(),
     * the inputting step through std::getline will be skipped.
     * Therefore, no matter using std::getline, it is better to add
     * std::cin.ignore() after using std::cin.
     */
    std::cin.ignore();

    if (number >= 1 && number - 1 < supportedProviders.size()) {
        selectedProvider = supportedProviders[number - 1];
    }

    return selectedProvider;
}

std::string Options::selectKey(
    const std::unordered_map<std::string, std::string>& keys, const std::string& purpose
    ) {
    std::string selectedKey;
    std::vector<std::string> providers;
    int i = 1;

    for (const auto& provider : std::views::keys(keys)) {
        std::cout << i << ". " << provider << "  ";
        providers.emplace_back(provider);
        ++i;
    }

    std::cout << "\n";

    std::cout << "Please input the number of the key you want to " << purpose << ".\n"
              << "(press 0 to exit without any selection)\n";

    int number;
    std::cin >> number;
    std::cin.ignore();

    // As the initial value of i is 1, the actual index should be i minus 1.
    if (number >= 1 && number - 1 < providers.size()) {
        selectedKey = providers[number - 1];
    }

    return selectedKey;
}

std::string Options::selectModel(
    const std::unordered_map<std::string, std::unordered_map<std::string, std::string>>& models,
    const std::string& purpose
    ) {
    std::string selectedModel;
    std::vector<std::string> listedNames;
    int i = 1;

    auto supportedProviders = Config::supportedProviders;

    for (const auto& provider : supportedProviders) {
        if (models.contains(provider)) {
            for (const auto& [customName, identifierName] : models.at(provider)) {
                std::cout << i << ". " << customName << " : " << identifierName << "  ";

                std::string listedName;
                listedName += provider;
                listedName += ":";
                listedName += customName;
                listedNames.emplace_back(listedName);

                ++i;
            }
        }
    }

    std::cout << "\n";

    std::cout << "Please input the number of the model you want to " << purpose << ".\n"
              << "(press 0 to exit without any selection)\n";

    int number;
    std::cin >> number;
    std::cin.ignore();

    // As the initial value of i is 1, the actual index should be i minus 1.
    if (number >= 1 && number - 1 < listedNames.size()) {
        selectedModel = listedNames[number - 1];
    }

    return selectedModel;
}

std::string Options::selectRole(
    const std::unordered_map<std::string, std::array<std::string, 2>>& roles, const std::string& purpose
    ) {
    std::string selectedRole;
    std::vector<std::string> roleNames;
    int i = 1;

    for (const auto& roleName : std::views::keys(roles)) {
        std::cout << i << ". " << roleName << "  ";
        roleNames.emplace_back(roleName);
        ++i;
    }

    std::cout << "\n";

    std::cout << "Please input the number of the role you want to " << purpose << ".\n"
              << "(press 0 to exit without any selection)\n";

    int number;
    std::cin >> number;
    std::cin.ignore();

    // As the initial value of i is 1, the actual index should be i minus 1.
    if (number >= 1 && number - 1 < roleNames.size()) {
        selectedRole = roleNames[number - 1];
    }

    return selectedRole;
}

std::expected<std::array<std::string, 6>, std::string>
Options::getOptions(int argc, char** argv) {
    // All available options.
    std::vector<std::string> options = {
        "--add-key",  // Add or update an API key.
        "--add-model",  // Add or update an AI model.
        "--add-role",  // Add or update a preset role with instructions.
        "--show-keys",
        "--show-models",
        "--show-roles",
        "--show-role",
        "--remove-key",
        "--remove-model",
        "--remove-role",
        "-model",  // Specify an AI model.
        "-role",  // Specify a preset role for AI, or give a role instruction to AI.
        "-content"  // Give detailed instructions to AI.
    };

    std::string connectionProvider;
    std::string connectionKey;
    std::string connectionModel;
    std::string connectionRoleInstruction;
    std::string connectionDetailedInstructions;
    std::string connectionContent;

    std::vector<std::string> inputtedArguments;

    /* Collect all arguments.
     * As argv[0] is the program's name, i begins at 1.
     */
    for (int i = 1; i < argc; ++i) {
        inputtedArguments.emplace_back(argv[i]);
    }

    // Recognize options.
    for (std::string& option: options) {
        /* The options which names are not inputted will not be processed.
         * So here they are directly ignored.
         * By contrast, if a user inputs an option's name but does not specify any value,
         * the program should ask about the value step by step.
         */
        if (!std::ranges::contains(inputtedArguments, option)) {
            continue;
        }

        /* Firstly, "drop_while" removes all contents before a valid option's name.
         * Secondly, "drop" removes the option's name.
         * Thirdly, "take_while" removes all contents starting with another valid option's name.
         */
        auto viewVector =
            inputtedArguments
            | std::views::drop_while(
                [option](auto& argument) {
                    return argument != option;
                }
              )
            | std::views::drop(1)
            | std::views::take_while(
                [options](auto& argument) {
                    return !std::ranges::contains(options, argument);
                }
              )
            | std::ranges::to<std::vector<std::string>>();

        /* The value of an option may be split to multiple vectors because of the mechanism
         * that a terminal handles spaces. The vectors should be combined into a string for
         * further processing.
         */
        auto optionValue =
            std::views::join(viewVector)
            | std::ranges::to<std::string>();

        auto supportedProviders = Config::supportedProviders;

        if (option == "--add-key") {
            if (optionValue.empty()) {
                auto selectedProvider = selectProvider("the API key");

                if (!selectedProvider.empty()) {
                    const auto& providerName = selectedProvider;

                    std::cout << "Please input the API key: ";
                    std::string key;
                    std::getline(std::cin, key);

                    if (!key.empty()) {
                        optionValue += providerName;
                        optionValue += ":";
                        optionValue += key;
                    }
                }
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The API key is saved." << std::endl;
                }
            }
        }

        else if (option == "--add-model") {
            if (optionValue.empty()) {
                auto selectedProvider = selectProvider("the model");

                if (!selectedProvider.empty()) {
                    const auto& providerName = selectedProvider;

                    std::cout << "Please input the custom name of the AI model. "
                                 "You may give a name that is easy to remember.\n";
                    std::string customName;
                    std::getline(std::cin, customName);

                    /* After using std::getline, it does not need to use std::cin.ignore().
                     * The method std::getline discards \n, so what std::cin.ignore() skips
                     * is the first character of the new line, and it makes the new line lose
                     * the first character.
                     */
                    std::cout << "Please input the identifier name of the AI model. "
                                 "The identifier name will be used for API connections.\n";

                    std::string identifierName;
                    std::getline(std::cin, identifierName);

                    if (!customName.empty() && !identifierName.empty()) {
                        optionValue += providerName;
                        optionValue += ":";
                        optionValue += customName;
                        optionValue += ":";
                        optionValue += identifierName;
                    }
                }
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The AI model is saved." << std::endl;
                }
            }
        }

        else if (option == "--add-role") {
            if (optionValue.empty()) {
                std::cout << "Please set a name for the preset role: ";
                std::string roleName;
                std::getline(std::cin, roleName);

                std::cout << "Please write a role instruction:\n";
                std::string roleInstruction;
                std::getline(std::cin, roleInstruction);

                std::cout << "Please write detailed instructions for the role.\n"
                             "You may input multiple lines and use the Enter key to start a new line.\n"
                             "For exiting without inputting anything, just press the Enter key.\n"
                             "If you have inputted something, please press the Enter key twice for exiting.\n";
                std::string detailedInstructions;
                std::string detailedInstructionsLine;
                while (std::getline(std::cin, detailedInstructionsLine)) {
                    if (detailedInstructionsLine.empty()) {
                        break;
                    }

                    detailedInstructions += detailedInstructionsLine;
                    detailedInstructions += '\n';
                }

                if (!roleName.empty() && !detailedInstructions.empty()) {
                    optionValue += roleName;
                    optionValue += ":";
                    optionValue += roleInstruction;
                    optionValue += ":";
                    optionValue += detailedInstructions;
                }
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The role is saved." << std::endl;
                }
            }
        }

        else if (option == "--show-keys") {
            auto keys = std::get<0>(Config::loadConf("read"));

            if (keys.empty()) {
                std::cout << "No key is found." << std::endl;
                continue;
            }

            for (auto& provider : supportedProviders) {
                if (keys.contains(provider)) {
                    std::string hiddenKey;
                    auto key = keys[provider];
                    int i = 0;
                    while (i < key.size()) {
                        if (i < key.size() - 4) {
                            hiddenKey += '*';
                        } else {
                            hiddenKey += key[i];
                        }

                        ++i;
                    }

                    std::cout << provider << " : " << hiddenKey << std::endl;
                }
            }
        }

        else if (option == "--show-models") {
            auto models = std::get<1>(Config::loadConf("read"));

            if (models.empty()) {
                std::cout << "No model is found." << std::endl;
                continue;
            }

            for (auto& provider : supportedProviders) {
                if (models.contains(provider)) {
                    std::cout << "=== " << provider << " models ===\n";

                    for (const auto& [customName, identifierName] : models[provider]) {
                        std::cout << customName << " : " << identifierName << std::endl;
                    }

                    std::cout << "\n";
                }
            }
        }

        else if (option == "--show-roles") {
            auto roles = std::get<2>(Config::loadConf("read"));

            if (roles.empty()) {
                std::cout << "No preset role is found." << std::endl;
                continue;
            }

            for (const auto& roleName : std::views::keys(roles)) {
                std::cout << roleName << std::endl;
            }
        }

        else if (option == "--show-role") {
            auto roles = std::get<2>(Config::loadConf("read"));

            if (roles.empty()) {
                std::cout << "No preset role is found." << std::endl;
                continue;
            }

            if (optionValue.empty()) {
                optionValue = selectRole(roles, "view");
            }

            for (const auto& [roleName, roleInstructions] : roles) {
                if (optionValue == roleName) {
                    std::cout << roleName << std::endl;
                    std::cout << "Role instruction:\n" << roleInstructions[0] << std::endl;
                    std::cout << "Detailed instructions:\n" << roleInstructions[1] << std::endl;
                }
            }
        }

        else if (option == "--remove-key") {
            auto keys = std::get<0>(Config::loadConf("read"));

            if (keys.empty()) {
                std::cout << "No key is found." << std::endl;
                continue;
            }

            if (optionValue.empty()) {
                optionValue = selectKey(keys, "remove");
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The key is removed." << std::endl;
                }
            }
        }

        else if (option == "--remove-model") {
            auto models = std::get<1>(Config::loadConf("read"));

            if (models.empty()) {
                std::cout << "No model is found." << std::endl;
                continue;
            }

            if (optionValue.empty()) {
                optionValue = selectModel(models, "remove");
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The model is removed." << std::endl;
                }
            }
        }

        else if (option == "--remove-role") {
            auto roles = std::get<2>(Config::loadConf("read"));

            if (roles.empty()) {
                std::cout << "No preset role is found." << std::endl;
                continue;
            }

            if (optionValue.empty()) {
                optionValue = selectRole(roles, "remove");
            }

            if (!optionValue.empty()) {
                bool isSaved = Config::saveConf(option, optionValue);

                if (isSaved) {
                    std::cout << "\n" << "The role is removed." << std::endl;
                }
            }
        }

        else if (option == "-model") {
            auto models = std::get<1>(Config::loadConf("read"));

            if (models.empty()) {
                std::cout << "No model is found." << std::endl;
                continue;
            }

            if (optionValue.empty()) {
                optionValue = selectModel(models, "use");
            }

            if (optionValue.empty()) {
                continue;
            }

            auto modelVector = optionValue
            | std::views::split(':')
            | std::ranges::to<std::vector<std::string>>();

            std::string supposedProvider;
            std::string supposedCustomName;
            if (modelVector.size() == 1) {
                supposedCustomName = Config::cleanString(modelVector[0]);

                for (auto& [provider, modelNames] : models) {
                    for (auto& [customName, identifierName] : modelNames) {
                        if (supposedCustomName == customName) {
                            connectionProvider = provider;
                            connectionModel = identifierName;
                        }
                    }
                }
            }

            if (modelVector.size() >= 2) {
                supposedProvider = Config::cleanString(modelVector[0]);
                supposedCustomName = Config::cleanString(modelVector[1]);

                for (auto& [provider, modelNames] : models) {
                    for (auto& [customName, identifierName] : modelNames) {
                        if (supposedProvider == provider && supposedCustomName == customName) {
                            connectionProvider = provider;
                            connectionModel = identifierName;
                        }
                    }
                }
            }
        }

        else if (option == "-role") {
            auto roles = std::get<2>(Config::loadConf("read"));

            if (optionValue.empty() && !roles.empty()) {
                std::cout << "You may select a preset role.\n"
                             "If you want to give a custom role instruction, please press 0.\n";

                auto selectedRoleName = selectRole(roles, "use");

                if (!selectedRoleName.empty()) {
                    for (auto& [roleName, roleInstructions] : roles) {
                        if (selectedRoleName == roleName) {
                            connectionRoleInstruction = roleInstructions[0];
                            connectionDetailedInstructions = roleInstructions[1];
                        }
                    }
                }
            }

            if (!connectionRoleInstruction.empty() && !connectionDetailedInstructions.empty()) {
                continue;
            }

            std::cout << "Please give a role instruction for AI:" << std::endl;
            std::string roleInstruction;
            std::getline(std::cin, roleInstruction);

            if (!roleInstruction.empty()) {
                optionValue = roleInstruction;
            }

            if (!optionValue.empty()) {
                connectionRoleInstruction = Config::cleanString(optionValue);
            }
        }

        else if (option == "-content") {
            if (optionValue.empty()) {
                std::cout << "Please give specific instructions for AI.\n"
                             "You may input multiple lines and use the Enter key to start a new line.\n"
                             "For exiting without inputting anything, just press the Enter key.\n"
                             "If you have inputted something, please press the Enter key twice for exiting.\n";
                std::string supposedContent;

                while (std::getline(std::cin, supposedContent)) {
                    if (supposedContent.empty()) {
                        break;
                    }

                    optionValue += supposedContent;
                    optionValue += '\n';
                }
            }

            if (optionValue.empty()) {
                continue;
            }

            connectionContent = Config::cleanString(optionValue);
        }
    }

    auto keys = std::get<0>(Config::loadConf("read"));

    if (!connectionProvider.empty() && !keys.empty()) {
        for (auto& [provider, key] : keys) {
            if (connectionProvider == provider) {
                connectionKey = key;
            }
        }
    }

    if (connectionKey.empty()) {
        return std::unexpected("No API connection is performed, because no key is found.");
    }

    if (connectionModel.empty()) {
        return std::unexpected("No API connection is performed, because no model is found.");
    }

    if (connectionContent.empty()) {
        return std::unexpected("No API connection is performed, because there is no specific instructions.");
    }

    return std::array{
        connectionProvider,
        connectionKey,
        connectionModel,
        connectionRoleInstruction,
        connectionDetailedInstructions,
        connectionContent
    };
}